/***************************************************************************
 *
 * Copyright (c) 2000,2001,2002 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: misc.c,v 1.19.2.7 2003/07/07 17:31:49 sasa Exp $
 *
 * Author  : Bazsi, SaSa, Chaoron
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/zorplib.h>
#include <zorp/misc.h>
#include <zorp/log.h>

#include <string.h>
#include <stdlib.h>
#include <glib.h>
#ifndef G_OS_WIN32
  #include <crypt.h>
#else
  #include <openssl/des.h>
  #define crypt DES_crypt
#endif

#define PARSE_STATE_START  0
#define PARSE_STATE_DASH   1
#define PARSE_STATE_END    2
#define PARSE_STATE_ESCAPE 3

void 
z_charset_init(ZCharSet *self)
{
  memset(self, 0, sizeof(*self));
}

gboolean 
z_charset_parse(ZCharSet *self, gchar *interval_str)
{
  guint i = 0;
  guint j;
  guint state = PARSE_STATE_START;
  guint old_state = PARSE_STATE_START;
  guchar start_pos = 0;
  guchar end_pos = 0;
  
  z_enter();
  while (interval_str[i])
    {
      switch (state)
        {
        case PARSE_STATE_START:
          if (interval_str[i] == '\\')
            {
              z_cp();
              old_state = state;
              state = PARSE_STATE_ESCAPE;
            }
          else
            {
              z_cp();
              start_pos = interval_str[i];
              state = PARSE_STATE_DASH;
              i++;
            }
          break;
        case PARSE_STATE_DASH:
          if (interval_str[i] == '\\')
            {
              z_cp();
              state = PARSE_STATE_END;
              i--;
            }
          else
            {
              z_cp();
              state = PARSE_STATE_END;
              if (interval_str[i] == '-')
                i++;
              else
                i--;
            }
          break;
        case PARSE_STATE_END:
          if (interval_str[i] == '\\')
            {
              z_cp();
              old_state = state;
              state = PARSE_STATE_ESCAPE;
            }
          else
            {
              z_cp();
              end_pos = interval_str[i];
              for (j = start_pos; j <= end_pos; j++)
                z_charset_enable(self, j);
              
              i++;
              state = PARSE_STATE_START;
            }
          break;
        case PARSE_STATE_ESCAPE:
          z_cp();
          i++;
          state = old_state;
          break;
        default:
          z_leave();
          return FALSE;
        }
    }

  if (state == PARSE_STATE_DASH)
    {
      z_cp();
      z_charset_enable(self, start_pos);
      state = PARSE_STATE_START;
    }
  
  if (state == PARSE_STATE_START)
    {
      z_leave();
      return TRUE;
    }
  
  z_leave();
  return FALSE;
}

gboolean 
z_charset_is_string_valid(ZCharSet *self, gchar *str, gint len)
{
  gint i;
  
  if (len < 0)
    len = strlen(str);
  
  for (i = 0; i < len; i++)
    {
      if (!z_charset_is_enabled(self, str[i]))
        return FALSE;
    }
  return TRUE;
}


GString *
g_string_assign_len(GString *s, gchar *val, gint len)
{
  g_string_truncate(s, 0);
  if (val && len)
    g_string_append_len(s, val, len);
  return s;
}

void 
z_data_dump(char *session_id, const char *buf, guint len)
{
  guint i, j;
  gchar line[1024];
  
  i = 0;
  while (i < len)
    {
      char *end = line;
      
      for (j = 0; j < 16 && (i + j < len); j++)
        {
          g_snprintf(end, sizeof(line) - (end - line), "%02X ", (unsigned char) buf[i+j]);
          end += 3;
        }
      g_snprintf(end, sizeof(line) - (end - line), " ");
      end++;
      for (j = 0; j < 16 && (i + j < len) && sizeof(line) > (end - line); j++)
        {
          *end = buf[i + j] > 31 ? buf[i + j] : '.';
          end++;
        }
      *end='\0';
          
      i += j;
      
      /*NOLOG*/
      z_log(session_id, CORE_DUMP, 9, "data line: %s", line);
    }
}

gchar *
z_str_escape(const gchar *s, gint len)
{
  gchar *res;
  gint i = 0, j = 0;;
  
  z_enter();
  
  if (len < 0)
    len = strlen(s) + 1;
  
  res = g_new0(gchar, len * 2);
  
  while (i < len && s[i] != '\0')
    {
      switch (s[i])
        {
	  case ' ':
	    res[j++] = '%';
            res[j++] = '_';
	    break;
          
	  case '%':
	    res[j++] = '%';
	    res[j++] = '%';
	    break;
	  
	  default:
	    res[j++] = s[i];
	}
      i++;
    }
  
  z_leave();
  return res;
}

gchar *
z_str_compress(const gchar *s, gint len)
{
  gchar *res;
  gint i = 0, j = 0;;
  
  z_enter();
  
  if (len < 0)
    len = strlen(s) + 1;
  
  res = g_new0(gchar, len);
  
  while (i < len && s[i] != '\0')
    {
      if (s[i] == '%' && s[i+1] == '%')
        {
	  i++;
	  res[j++] = '%';
	}
      else if (s[i] == '%' && s[i+1] == '_')
        {
	  i++;
	  res[j++] = ' ';
	}
      else
        {
	  res[j++] = s[i];
	}
      i++;
    }
  
  z_leave();
  return res;
}

gboolean
z_port_enabled(gchar *port_range, guint port)
{
  long int portl, porth;
  gchar *tmp;
  gchar *err;
  
  if (strlen(port_range) == 0 )
    return FALSE;
  
  tmp = port_range;
  while (*tmp)
    {
      portl = strtol(tmp, &err, 10);
      tmp = err;
      if (*tmp == '-')
        {
          porth = strtol(tmp, &err, 10);
          tmp = err;
        }
      else
        {
          porth = portl;
        }

      if (*tmp != 0 &&  *tmp != ',')
        return FALSE;
    
      if (*tmp)
        {
          tmp++;
          if (*tmp <= '0' && *tmp >= '9')
          return FALSE;
        }
        
      if ( portl <= (long int)port && (long int)port <= porth )
        return TRUE;

    }
  return FALSE;
}


/* mutex protecting crypt() invocation */
static GStaticMutex crypt_lock = G_STATIC_MUTEX_INIT;

void
z_crypt(const char *key, const char *salt, char *result, size_t result_len)
{
  g_static_mutex_lock(&crypt_lock);
  g_strlcpy(result, crypt(key, salt), result_len);
  g_static_mutex_unlock(&crypt_lock);
}

#define ON_OFF_STR(x) (x ? "on" : "off")

gchar *
z_zorplib_version_info(void)
{
  static gchar buf[128];
  
  g_snprintf(buf, sizeof(buf),
             "Zorplib %s\n"
             "Compile-Date: %s %s\n"
             "Trace: %s\n"
             "MemTrace: %s\n"
             "Caps: %s\n"
             "Debug: %s\n"
             "StackDump: %s\n",

             ZORPLIBLL_VERSION, __DATE__, __TIME__,
             ON_OFF_STR(ZORPLIB_ENABLE_TRACE),
             ON_OFF_STR(ZORPLIB_ENABLE_MEM_TRACE),
             ON_OFF_STR(ZORPLIB_ENABLE_CAPS),
             ON_OFF_STR(ZORPLIB_ENABLE_DEBUG),
             ON_OFF_STR(ZORPLIB_ENABLE_STACKDUMP));
  return buf;
}
